/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.mavibot.btree;

import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.directory.mavibot.btree.AbstractPage;
import org.apache.directory.mavibot.btree.Addition;
import org.apache.directory.mavibot.btree.BTreeConfiguration;
import org.apache.directory.mavibot.btree.BTreeHeader;
import org.apache.directory.mavibot.btree.BTreeTypeEnum;
import org.apache.directory.mavibot.btree.Cursor;
import org.apache.directory.mavibot.btree.DeleteResult;
import org.apache.directory.mavibot.btree.Deletion;
import org.apache.directory.mavibot.btree.DuplicateKeyMemoryHolder;
import org.apache.directory.mavibot.btree.ElementHolder;
import org.apache.directory.mavibot.btree.InsertResult;
import org.apache.directory.mavibot.btree.Leaf;
import org.apache.directory.mavibot.btree.MemoryHolder;
import org.apache.directory.mavibot.btree.Modification;
import org.apache.directory.mavibot.btree.ModifyResult;
import org.apache.directory.mavibot.btree.Node;
import org.apache.directory.mavibot.btree.NotPresentResult;
import org.apache.directory.mavibot.btree.Page;
import org.apache.directory.mavibot.btree.PoisonPill;
import org.apache.directory.mavibot.btree.RecordManager;
import org.apache.directory.mavibot.btree.ReferenceHolder;
import org.apache.directory.mavibot.btree.RemoveResult;
import org.apache.directory.mavibot.btree.SplitResult;
import org.apache.directory.mavibot.btree.Transaction;
import org.apache.directory.mavibot.btree.Tuple;
import org.apache.directory.mavibot.btree.exception.KeyNotFoundException;
import org.apache.directory.mavibot.btree.serializer.BufferHandler;
import org.apache.directory.mavibot.btree.serializer.ElementSerializer;
import org.apache.directory.mavibot.btree.serializer.LongSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BTree<K, V> {
    protected static final Logger LOG = LoggerFactory.getLogger(BTree.class);
    private BTreeHeader btreeHeader;
    public static final int DEFAULT_PAGE_SIZE = 16;
    public static final int DEFAULT_WRITE_BUFFER_SIZE = 1024000;
    public static final String DEFAULT_JOURNAL = "mavibot.log";
    public static final String DATA_SUFFIX = ".db";
    public static final String JOURNAL_SUFFIX = ".log";
    private Comparator<K> comparator;
    protected volatile Page<K, V> rootPage;
    private ConcurrentLinkedQueue<Transaction<K, V>> readTransactions;
    private int writeBufferSize;
    protected Class<?> keyType;
    private ElementSerializer<K> keySerializer;
    private ElementSerializer<V> valueSerializer;
    private File file;
    private RecordManager recordManager;
    private BTreeTypeEnum type;
    private boolean withJournal;
    private File journal;
    private ReentrantLock writeLock;
    private Thread readTransactionsThread;
    private Thread journalManagerThread;
    public static final long DEFAULT_READ_TIMEOUT = 10000L;
    private long readTimeOut = 10000L;
    private BlockingQueue<Modification<K, V>> modificationsQueue;
    private File envDir;

    private void createTransactionManager() {
        Runnable readTransactionTask = new Runnable(){

            @Override
            public void run() {
                try {
                    Transaction transaction = null;
                    while (!Thread.currentThread().isInterrupted()) {
                        long timeoutDate = System.currentTimeMillis() - BTree.this.readTimeOut;
                        long t0 = System.currentTimeMillis();
                        int nbTxns = 0;
                        while ((transaction = (Transaction)BTree.this.readTransactions.peek()) != null) {
                            ++nbTxns;
                            if (transaction.isClosed()) {
                                BTree.this.readTransactions.poll();
                                continue;
                            }
                            if (transaction.getCreationDate() >= timeoutDate) break;
                            transaction.close();
                            BTree.this.readTransactions.poll();
                        }
                        long t1 = System.currentTimeMillis();
                        if (nbTxns > 0) {
                            System.out.println("Processing old txn : " + nbTxns + ", " + (t1 - t0) + "ms");
                        }
                        Thread.sleep(BTree.this.readTimeOut);
                    }
                }
                catch (InterruptedException ie) {
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
        };
        this.readTransactionsThread = new Thread(readTransactionTask);
        this.readTransactionsThread.setDaemon(true);
        this.readTransactionsThread.start();
    }

    private void createJournalManager() {
        Runnable journalTask = new Runnable(){

            private boolean flushModification(FileChannel channel, Modification<K, V> modification) throws IOException {
                if (modification instanceof Addition) {
                    byte[] keyBuffer = BTree.this.keySerializer.serialize(modification.getKey());
                    ByteBuffer bb = ByteBuffer.allocateDirect(keyBuffer.length + 1);
                    bb.put((byte)0);
                    bb.put(keyBuffer);
                    bb.flip();
                    channel.write(bb);
                    byte[] valueBuffer = BTree.this.valueSerializer.serialize(modification.getValue());
                    bb = ByteBuffer.allocateDirect(valueBuffer.length);
                    bb.put(valueBuffer);
                    bb.flip();
                    channel.write(bb);
                } else if (modification instanceof Deletion) {
                    byte[] keyBuffer = BTree.this.keySerializer.serialize(modification.getKey());
                    ByteBuffer bb = ByteBuffer.allocateDirect(keyBuffer.length + 1);
                    bb.put((byte)1);
                    bb.put(keyBuffer);
                    bb.flip();
                    channel.write(bb);
                } else {
                    return false;
                }
                channel.force(true);
                return true;
            }

            @Override
            public void run() {
                Modification modification = null;
                FileChannel channel = null;
                try {
                    boolean stop;
                    FileOutputStream stream = new FileOutputStream(BTree.this.journal);
                    channel = stream.getChannel();
                    while (!Thread.currentThread().isInterrupted() && !(stop = this.flushModification(channel, modification = (Modification)BTree.this.modificationsQueue.take()))) {
                    }
                }
                catch (InterruptedException ie) {
                    while ((modification = (Modification)BTree.this.modificationsQueue.peek()) != null) {
                    }
                    try {
                        this.flushModification(channel, modification);
                    }
                    catch (IOException ioe) {}
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
        };
        this.journalManagerThread = new Thread(journalTask);
        this.journalManagerThread.setDaemon(true);
        this.journalManagerThread.start();
    }

    public BTree() {
        this.btreeHeader = new BTreeHeader();
        this.type = BTreeTypeEnum.MANAGED;
    }

    public BTree(BTreeConfiguration<K, V> configuration) throws IOException {
        String name = configuration.getName();
        if (name == null) {
            throw new IllegalArgumentException("BTree name cannot be null");
        }
        String filePath = configuration.getFilePath();
        if (filePath != null) {
            this.envDir = new File(filePath);
        }
        this.btreeHeader = new BTreeHeader();
        this.btreeHeader.setName(name);
        this.btreeHeader.setPageSize(configuration.getPageSize());
        this.keySerializer = configuration.getKeySerializer();
        this.btreeHeader.setKeySerializerFQCN(this.keySerializer.getClass().getName());
        this.valueSerializer = configuration.getValueSerializer();
        this.btreeHeader.setValueSerializerFQCN(this.valueSerializer.getClass().getName());
        this.comparator = this.keySerializer.getComparator();
        this.readTimeOut = configuration.getReadTimeOut();
        this.writeBufferSize = configuration.getWriteBufferSize();
        this.btreeHeader.setAllowDuplicates(configuration.isAllowDuplicates());
        this.type = configuration.getType();
        if (this.comparator == null) {
            throw new IllegalArgumentException("Comparator should not be null");
        }
        this.rootPage = new Leaf(this);
        this.init();
    }

    public BTree(String name, ElementSerializer<K> keySerializer, ElementSerializer<V> valueSerializer) throws IOException {
        this(name, keySerializer, valueSerializer, false);
    }

    public BTree(String name, ElementSerializer<K> keySerializer, ElementSerializer<V> valueSerializer, boolean allowDuplicates) throws IOException {
        this(name, null, keySerializer, valueSerializer, 16, allowDuplicates);
    }

    public BTree(String name, ElementSerializer<K> keySerializer, ElementSerializer<V> valueSerializer, int pageSize) throws IOException {
        this(name, null, keySerializer, valueSerializer, pageSize);
    }

    public BTree(String name, String path, ElementSerializer<K> keySerializer, ElementSerializer<V> valueSerializer) throws IOException {
        this(name, path, keySerializer, valueSerializer, 16);
    }

    public BTree(String name, String dataDir, ElementSerializer<K> keySerializer, ElementSerializer<V> valueSerializer, int pageSize) throws IOException {
        this(name, dataDir, keySerializer, valueSerializer, pageSize, false);
    }

    public BTree(String name, String dataDir, ElementSerializer<K> keySerializer, ElementSerializer<V> valueSerializer, int pageSize, boolean allowDuplicates) throws IOException {
        this.btreeHeader = new BTreeHeader();
        this.btreeHeader.setName(name);
        if (dataDir != null) {
            this.envDir = new File(dataDir);
        }
        this.setPageSize(pageSize);
        this.writeBufferSize = 1024000;
        this.keySerializer = keySerializer;
        this.btreeHeader.setKeySerializerFQCN(keySerializer.getClass().getName());
        this.valueSerializer = valueSerializer;
        this.btreeHeader.setValueSerializerFQCN(valueSerializer.getClass().getName());
        this.comparator = keySerializer.getComparator();
        this.btreeHeader.setAllowDuplicates(allowDuplicates);
        this.rootPage = new Leaf(this);
        this.init();
    }

    public void init() throws IOException {
        if (this.envDir != null && this.type != BTreeTypeEnum.MANAGED) {
            boolean created;
            if (!this.envDir.exists() && !(created = this.envDir.mkdirs())) {
                throw new IllegalStateException("Could not create the directory " + this.envDir + " for storing data");
            }
            this.file = new File(this.envDir, this.btreeHeader.getName() + DATA_SUFFIX);
            this.journal = new File(this.envDir, this.file.getName() + JOURNAL_SUFFIX);
            this.type = BTreeTypeEnum.PERSISTENT;
        }
        this.readTransactions = new ConcurrentLinkedQueue();
        Class<?> comparatorClass = this.comparator.getClass();
        Type[] types = comparatorClass.getGenericInterfaces();
        if (types[0] instanceof Class) {
            this.keyType = (Class)types[0];
        } else {
            Type[] argumentTypes = ((ParameterizedType)types[0]).getActualTypeArguments();
            if (argumentTypes != null && argumentTypes.length > 0 && argumentTypes[0] instanceof Class) {
                this.keyType = (Class)argumentTypes[0];
            }
        }
        this.writeLock = new ReentrantLock();
        if (this.type == BTreeTypeEnum.PERSISTENT) {
            this.modificationsQueue = new LinkedBlockingDeque<Modification<K, V>>();
            if (this.file.length() > 0L) {
                this.load(this.file);
            }
            this.withJournal = true;
            if (this.journal.length() > 0L) {
                this.applyJournal();
            }
            this.createJournalManager();
        } else if (this.type == null) {
            this.type = BTreeTypeEnum.IN_MEMORY;
        }
    }

    public void close() throws IOException {
        if (this.type == BTreeTypeEnum.PERSISTENT) {
            this.modificationsQueue.add(new PoisonPill());
            this.flush();
        }
        this.rootPage = null;
    }

    long getBtreeOffset() {
        return this.btreeHeader.getBTreeOffset();
    }

    void setBtreeOffset(long btreeOffset) {
        this.btreeHeader.setBTreeOffset(btreeOffset);
    }

    long getRootPageOffset() {
        return this.btreeHeader.getRootPageOffset();
    }

    void setRootPageOffset(long rootPageOffset) {
        this.btreeHeader.setRootPageOffset(rootPageOffset);
    }

    long getNextBTreeOffset() {
        return this.btreeHeader.getNextBTreeOffset();
    }

    void setNextBTreeOffset(long nextBTreeOffset) {
        this.btreeHeader.setNextBTreeOffset(nextBTreeOffset);
    }

    private int getPowerOf2(int size) {
        int newSize = --size;
        newSize |= newSize >> 1;
        newSize |= newSize >> 2;
        newSize |= newSize >> 4;
        newSize |= newSize >> 8;
        newSize |= newSize >> 16;
        return ++newSize;
    }

    public void setPageSize(int pageSize) {
        if (pageSize <= 2) {
            this.btreeHeader.setPageSize(16);
        } else {
            this.btreeHeader.setPageSize(this.getPowerOf2(pageSize));
        }
    }

    void setRoot(Page<K, V> root) {
        this.rootPage = root;
    }

    RecordManager getRecordManager() {
        return this.recordManager;
    }

    void setRecordManager(RecordManager recordManager) {
        this.recordManager = recordManager;
        this.type = BTreeTypeEnum.MANAGED;
    }

    public int getPageSize() {
        return this.btreeHeader.getPageSize();
    }

    long generateRevision() {
        return this.btreeHeader.incrementRevision();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public V insert(K key, V value) throws IOException {
        long revision = this.generateRevision();
        V existingValue = null;
        try {
            this.writeLock.lock();
            InsertResult<K, V> result = this.insert(key, value, revision);
            if (result instanceof ModifyResult) {
                existingValue = ((ModifyResult)result).getModifiedValue();
            }
        }
        finally {
            this.writeLock.unlock();
        }
        return existingValue;
    }

    public Tuple<K, V> delete(K key) throws IOException {
        if (key == null) {
            throw new IllegalArgumentException("Key must not be null");
        }
        long revision = this.generateRevision();
        Tuple<K, V> deleted = this.delete(key, revision);
        return deleted;
    }

    public Tuple<K, V> delete(K key, V value) throws IOException {
        if (key == null) {
            throw new IllegalArgumentException("Key must not be null");
        }
        if (value == null) {
            throw new IllegalArgumentException("Value must not be null");
        }
        long revision = this.generateRevision();
        Tuple<K, V> deleted = this.delete(key, value, revision);
        return deleted;
    }

    private Tuple<K, V> delete(K key, long revision) throws IOException {
        return this.delete(key, null, revision);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Tuple<K, V> delete(K key, V value, long revision) throws IOException {
        this.writeLock.lock();
        try {
            Tuple tuple = null;
            DeleteResult<K, V> result = this.rootPage.delete(revision, key, value, null, -1);
            if (result instanceof NotPresentResult) {
                Tuple<K, V> tuple2 = null;
                return tuple2;
            }
            Page<K, V> oldRootPage = this.rootPage;
            if (result instanceof RemoveResult) {
                RemoveResult removeResult = (RemoveResult)result;
                Page modifiedPage = removeResult.getModifiedPage();
                if (this.isManaged()) {
                    ElementHolder holder = this.recordManager.writePage(this, modifiedPage, revision);
                    ((AbstractPage)modifiedPage).setOffset(((ReferenceHolder)holder).getOffset());
                    ((AbstractPage)modifiedPage).setLastOffset(((ReferenceHolder)holder).getLastOffset());
                }
                this.rootPage = modifiedPage;
                tuple = removeResult.getRemovedElement();
            }
            if (this.type == BTreeTypeEnum.PERSISTENT) {
                this.modificationsQueue.add(new Deletion(key));
            }
            if (tuple != null) {
                this.btreeHeader.decrementNbElems();
                if (this.isManaged()) {
                    this.recordManager.updateBtreeHeader(this, ((AbstractPage)this.rootPage).getOffset());
                }
            }
            if (this.isManaged()) {
                this.recordManager.addFreePages(this, result.getCopiedPages());
                this.recordManager.storeRootPage(this, this.rootPage);
            }
            Tuple tuple3 = tuple;
            return tuple3;
        }
        finally {
            this.writeLock.unlock();
        }
    }

    public V get(K key) throws IOException, KeyNotFoundException {
        return this.rootPage.get(key);
    }

    public BTree<V, V> getValues(K key) throws IOException, KeyNotFoundException {
        return this.rootPage.getValues(key);
    }

    public V get(long revision, K key) throws IOException, KeyNotFoundException {
        Page<K, V> revisionRootPage = this.getRootPage(revision);
        return revisionRootPage.get(key);
    }

    public boolean hasKey(K key) throws IOException {
        if (key == null) {
            return false;
        }
        return this.rootPage.hasKey(key);
    }

    public boolean hasKey(long revision, K key) throws IOException, KeyNotFoundException {
        if (key == null) {
            return false;
        }
        Page<K, V> revisionRootPage = this.getRootPage(revision);
        return revisionRootPage.hasKey(key);
    }

    public boolean contains(K key, V value) throws IOException {
        return this.rootPage.contains(key, value);
    }

    public boolean contains(long revision, K key, V value) throws IOException, KeyNotFoundException {
        Page<K, V> revisionRootPage = this.getRootPage(revision);
        return revisionRootPage.contains(key, value);
    }

    public Cursor<K, V> browse() throws IOException {
        Transaction<K, V> transaction = this.beginReadTransaction();
        LinkedList stack = new LinkedList();
        Cursor<K, V> cursor = this.rootPage.browse(transaction, stack);
        return cursor;
    }

    public Cursor<K, V> browse(long revision) throws IOException, KeyNotFoundException {
        Transaction<K, V> transaction = this.beginReadTransaction();
        Page<K, V> revisionRootPage = this.getRootPage(revision);
        LinkedList stack = new LinkedList();
        Cursor<K, V> cursor = revisionRootPage.browse(transaction, stack);
        return cursor;
    }

    public Cursor<K, V> browseFrom(K key) throws IOException {
        Transaction<K, V> transaction = this.beginReadTransaction();
        Cursor<K, V> cursor = this.rootPage.browse(key, transaction, new LinkedList());
        return cursor;
    }

    public Cursor<K, V> browseFrom(long revision, K key) throws IOException, KeyNotFoundException {
        Transaction<K, V> transaction = this.beginReadTransaction();
        Page<K, V> revisionRootPage = this.getRootPage(revision);
        LinkedList stack = new LinkedList();
        Cursor<K, V> cursor = revisionRootPage.browse(key, transaction, stack);
        return cursor;
    }

    InsertResult<K, V> insert(K key, V value, long revision) throws IOException {
        if (key == null) {
            throw new IllegalArgumentException("Key must not be null");
        }
        Object modifiedValue = null;
        InsertResult<K, V> result = this.rootPage.insert(revision, key, value);
        if (result instanceof ModifyResult) {
            ModifyResult modifyResult = (ModifyResult)result;
            Page modifiedPage = modifyResult.getModifiedPage();
            if (this.isManaged()) {
                ElementHolder holder = this.recordManager.writePage(this, modifiedPage, revision);
                ((AbstractPage)modifiedPage).setOffset(((ReferenceHolder)holder).getOffset());
                ((AbstractPage)modifiedPage).setLastOffset(((ReferenceHolder)holder).getLastOffset());
            }
            this.rootPage = modifiedPage;
            modifiedValue = modifyResult.getModifiedValue();
        } else {
            SplitResult splitResult = (SplitResult)result;
            Object pivot = splitResult.getPivot();
            Page leftPage = splitResult.getLeftPage();
            Page rightPage = splitResult.getRightPage();
            Node newRootPage = null;
            if (this.isManaged()) {
                ElementHolder holderLeft = this.recordManager.writePage(this, leftPage, revision);
                ((AbstractPage)splitResult.getLeftPage()).setOffset(((ReferenceHolder)holderLeft).getOffset());
                ((AbstractPage)splitResult.getLeftPage()).setLastOffset(((ReferenceHolder)holderLeft).getLastOffset());
                ElementHolder holderRight = this.recordManager.writePage(this, rightPage, revision);
                ((AbstractPage)splitResult.getRightPage()).setOffset(((ReferenceHolder)holderRight).getOffset());
                ((AbstractPage)splitResult.getRightPage()).setLastOffset(((ReferenceHolder)holderRight).getLastOffset());
                newRootPage = new Node(this, revision, pivot, holderLeft, holderRight);
            } else {
                newRootPage = new Node(this, revision, pivot, leftPage, rightPage);
            }
            if (this.isManaged()) {
                ElementHolder holder = this.recordManager.writePage(this, newRootPage, revision);
                ((AbstractPage)newRootPage).setOffset(((ReferenceHolder)holder).getOffset());
                ((AbstractPage)newRootPage).setLastOffset(((ReferenceHolder)holder).getLastOffset());
            }
            this.rootPage = newRootPage;
        }
        if (this.type == BTreeTypeEnum.PERSISTENT) {
            this.modificationsQueue.add(new Addition<K, V>(key, value));
        }
        if (modifiedValue == null) {
            this.btreeHeader.incrementNbElems();
        }
        if (this.isManaged()) {
            this.recordManager.updateBtreeHeader(this, ((AbstractPage)this.rootPage).getOffset());
            this.recordManager.addFreePages(this, result.getCopiedPages());
            this.recordManager.storeRootPage(this, this.rootPage);
        }
        return result;
    }

    private Transaction<K, V> beginReadTransaction() {
        Transaction<K, V> readTransaction = new Transaction<K, V>(this.rootPage, this.btreeHeader.getRevision() - 1L, System.currentTimeMillis());
        this.readTransactions.add(readTransaction);
        return readTransaction;
    }

    Class<?> getKeyType() {
        return this.keyType;
    }

    public Comparator<K> getComparator() {
        return this.comparator;
    }

    public void setComparator(Comparator<K> comparator) {
        this.comparator = comparator;
    }

    public void setKeySerializer(ElementSerializer<K> keySerializer) {
        this.keySerializer = keySerializer;
        this.comparator = keySerializer.getComparator();
        this.btreeHeader.setKeySerializerFQCN(keySerializer.getClass().getName());
    }

    public void setValueSerializer(ElementSerializer<V> valueSerializer) {
        this.valueSerializer = valueSerializer;
        this.btreeHeader.setValueSerializerFQCN(valueSerializer.getClass().getName());
    }

    private void writeBuffer(FileChannel channel, ByteBuffer bb, byte[] buffer) throws IOException {
        int size = buffer.length;
        int pos = 0;
        do {
            if (bb.remaining() >= size) {
                bb.put(buffer, pos, size);
                size = 0;
                continue;
            }
            int len = bb.remaining();
            size -= len;
            bb.put(buffer, pos, len);
            pos += len;
            bb.flip();
            channel.write(bb);
            bb.clear();
        } while (size > 0);
    }

    public void flush(File file) throws IOException {
        File parentFile = file.getParentFile();
        File baseDirectory = null;
        baseDirectory = parentFile != null ? new File(file.getParentFile().getAbsolutePath()) : new File(".");
        File tmpFileFD = File.createTempFile("mavibot", null, baseDirectory);
        FileOutputStream stream = new FileOutputStream(tmpFileFD);
        FileChannel ch = stream.getChannel();
        ByteBuffer bb = ByteBuffer.allocateDirect(this.writeBufferSize);
        Cursor<K, V> cursor = this.browse();
        if (this.keySerializer == null) {
            throw new RuntimeException("Cannot flush the btree without a Key serializer");
        }
        if (this.valueSerializer == null) {
            throw new RuntimeException("Cannot flush the btree without a Value serializer");
        }
        bb.putLong(this.btreeHeader.getNbElems());
        while (cursor.hasNext()) {
            Tuple<K, V> tuple = cursor.next();
            byte[] keyBuffer = this.keySerializer.serialize(tuple.getKey());
            this.writeBuffer(ch, bb, keyBuffer);
            byte[] valueBuffer = this.valueSerializer.serialize(tuple.getValue());
            this.writeBuffer(ch, bb, valueBuffer);
        }
        if (bb.position() > 0) {
            bb.flip();
            ch.write(bb);
        }
        ch.force(true);
        ch.close();
        File backupFile = File.createTempFile("mavibot", null, baseDirectory);
        file.renameTo(backupFile);
        tmpFileFD.renameTo(file);
        backupFile.delete();
    }

    private void applyJournal() throws IOException {
        long revision = this.generateRevision();
        if (!this.journal.exists()) {
            throw new IOException("The journal does not exist");
        }
        FileChannel channel = new RandomAccessFile(this.journal, "rw").getChannel();
        ByteBuffer buffer = ByteBuffer.allocate(65536);
        BufferHandler bufferHandler = new BufferHandler(channel, buffer);
        try {
            while (true) {
                K key;
                byte[] type;
                if ((type = bufferHandler.read(1))[0] == 0) {
                    key = this.keySerializer.deserialize(bufferHandler);
                    V value = this.valueSerializer.deserialize(bufferHandler);
                    this.insert(key, value, revision);
                    continue;
                }
                key = this.keySerializer.deserialize(bufferHandler);
                this.delete(key, revision);
            }
        }
        catch (EOFException eofe) {
            this.journal.delete();
            this.journal.createNewFile();
            return;
        }
    }

    public void load(File file) throws IOException {
        long revision = this.generateRevision();
        if (!file.exists()) {
            throw new IOException("The file does not exist");
        }
        FileChannel channel = new RandomAccessFile(file, "rw").getChannel();
        ByteBuffer buffer = ByteBuffer.allocate(65536);
        BufferHandler bufferHandler = new BufferHandler(channel, buffer);
        long nbElems = LongSerializer.deserialize(bufferHandler.read(8));
        this.btreeHeader.setNbElems(nbElems);
        boolean isJournalActivated = this.withJournal;
        this.withJournal = false;
        for (long i = 0L; i < nbElems; ++i) {
            K key = this.keySerializer.deserialize(bufferHandler);
            V value = this.valueSerializer.deserialize(bufferHandler);
            this.insert(key, value, revision);
        }
        this.withJournal = isJournalActivated;
    }

    private Page<K, V> getRootPage(long revision) throws IOException, KeyNotFoundException {
        if (this.isManaged()) {
            return this.recordManager.getRootPage(this, revision);
        }
        return this.rootPage;
    }

    public void flush() throws IOException {
        if (this.type == BTreeTypeEnum.PERSISTENT) {
            this.flush(this.file);
            FileOutputStream stream = new FileOutputStream(this.journal);
            FileChannel channel = stream.getChannel();
            channel.position(0L);
            channel.force(true);
        }
    }

    public long getReadTimeOut() {
        return this.readTimeOut;
    }

    public void setReadTimeOut(long readTimeOut) {
        this.readTimeOut = readTimeOut;
    }

    public String getName() {
        return this.btreeHeader.getName();
    }

    public void setName(String name) {
        this.btreeHeader.setName(name);
    }

    public File getFile() {
        return this.file;
    }

    public File getJournal() {
        return this.journal;
    }

    public int getWriteBufferSize() {
        return this.writeBufferSize;
    }

    public void setWriteBufferSize(int writeBufferSize) {
        this.writeBufferSize = writeBufferSize;
    }

    public boolean isInMemory() {
        return this.type == BTreeTypeEnum.IN_MEMORY;
    }

    public boolean isPersistent() {
        return this.type == BTreeTypeEnum.IN_MEMORY;
    }

    public boolean isManaged() {
        return this.type == BTreeTypeEnum.MANAGED;
    }

    ElementHolder createHolder(Object value) {
        if (this.type == BTreeTypeEnum.MANAGED) {
            if (value instanceof Page) {
                return new ReferenceHolder(this, (Page)value, ((Page)value).getOffset(), ((Page)value).getLastOffset());
            }
            if (this.isAllowDuplicates()) {
                return new DuplicateKeyMemoryHolder(this, value);
            }
            return new MemoryHolder(this, value);
        }
        if (this.isAllowDuplicates() && !(value instanceof Page)) {
            return new DuplicateKeyMemoryHolder(this, value);
        }
        return new MemoryHolder(this, value);
    }

    public ElementSerializer<K> getKeySerializer() {
        return this.keySerializer;
    }

    public String getKeySerializerFQCN() {
        return this.btreeHeader.getKeySerializerFQCN();
    }

    public ElementSerializer<V> getValueSerializer() {
        return this.valueSerializer;
    }

    public String getValueSerializerFQCN() {
        return this.btreeHeader.getValueSerializerFQCN();
    }

    public long getRevision() {
        return this.btreeHeader.getRevision();
    }

    void setRevision(long revision) {
        this.btreeHeader.setRevision(revision);
    }

    public long getNbElems() {
        return this.btreeHeader.getNbElems();
    }

    void setNbElems(long nbElems) {
        this.btreeHeader.setNbElems(nbElems);
    }

    public boolean isAllowDuplicates() {
        return this.btreeHeader.isAllowDuplicates();
    }

    void setAllowDuplicates(boolean allowDuplicates) {
        this.btreeHeader.setAllowDuplicates(allowDuplicates);
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        switch (this.type) {
            case IN_MEMORY: {
                sb.append("In-memory ");
                break;
            }
            case MANAGED: {
                sb.append("Managed ");
                break;
            }
            case PERSISTENT: {
                sb.append("Persistent ");
            }
        }
        sb.append("BTree");
        sb.append("[").append(this.btreeHeader.getName()).append("]");
        sb.append("( pageSize:").append(this.btreeHeader.getPageSize());
        if (this.rootPage != null) {
            sb.append(", nbEntries:").append(this.btreeHeader.getNbElems());
        } else {
            sb.append(", nbEntries:").append(0);
        }
        sb.append(", comparator:");
        if (this.comparator == null) {
            sb.append("null");
        } else {
            sb.append(this.comparator.getClass().getSimpleName());
        }
        sb.append(", DuplicatesAllowed: ").append(this.btreeHeader.isAllowDuplicates());
        if (this.type == BTreeTypeEnum.PERSISTENT) {
            try {
                sb.append(", file : ");
                if (this.file != null) {
                    sb.append(this.file.getCanonicalPath());
                } else {
                    sb.append("Unknown");
                }
                sb.append(", journal : ");
                if (this.journal != null) {
                    sb.append(this.journal.getCanonicalPath());
                } else {
                    sb.append("Unkown");
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        sb.append(") : \n");
        sb.append(this.rootPage.dumpPage(""));
        return sb.toString();
    }
}

