/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.yarn;

import org.apache.flink.util.FlinkException;
import org.apache.flink.util.jackson.JacksonMapperFactory;

import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.core.JsonFactoryBuilder;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.core.JsonParser;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.core.JsonProcessingException;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.JsonNode;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.databind.node.ObjectNode;

import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.yarn.api.records.LocalResource;
import org.apache.hadoop.yarn.api.records.LocalResourceType;
import org.apache.hadoop.yarn.api.records.LocalResourceVisibility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Objects;

import static org.apache.flink.util.Preconditions.checkNotNull;

/**
 * Yarn local resource descriptor is generated by {@link YarnClusterDescriptor} and set to
 * environment of job manager. Then it will be used to register local resources for task manager in
 * {@link Utils#createTaskExecutorContext}.
 */
class YarnLocalResourceDescriptor {
    private static final Logger LOG = LoggerFactory.getLogger(YarnLocalResourceDescriptor.class);

    private static final ObjectMapper OBJECT_MAPPER =
            JacksonMapperFactory.createObjectMapper(
                            new JsonFactoryBuilder().quoteChar('\'').build())
                    .enable(JsonParser.Feature.ALLOW_SINGLE_QUOTES);

    private final String resourceKey;
    private final Path path;
    private final long size;
    private final long modificationTime;
    private final LocalResourceVisibility visibility;
    private final LocalResourceType resourceType;

    YarnLocalResourceDescriptor(
            String resourceKey,
            Path path,
            long resourceSize,
            long modificationTime,
            LocalResourceVisibility visibility,
            LocalResourceType resourceType) {
        this.resourceKey = checkNotNull(resourceKey);
        this.path = checkNotNull(path);
        this.size = resourceSize;
        this.modificationTime = modificationTime;
        this.visibility = checkNotNull(visibility);
        this.resourceType = checkNotNull(resourceType);
    }

    boolean alreadyRegisteredAsLocalResource() {
        return this.visibility.equals(LocalResourceVisibility.PUBLIC);
    }

    String getResourceKey() {
        return resourceKey;
    }

    Path getPath() {
        return path;
    }

    long getSize() {
        return size;
    }

    long getModificationTime() {
        return modificationTime;
    }

    LocalResourceVisibility getVisibility() {
        return visibility;
    }

    LocalResourceType getResourceType() {
        return resourceType;
    }

    static YarnLocalResourceDescriptor fromString(String desc) throws Exception {
        try {
            JsonNode node = OBJECT_MAPPER.readTree(desc);
            if (!validate(node)) {
                throw new FlinkException("Error to parse YarnLocalResourceDescriptor from " + desc);
            }
            return new YarnLocalResourceDescriptor(
                    node.get("resourceKey").asText(),
                    new Path(node.get("path").asText()),
                    node.get("size").asLong(),
                    node.get("modificationTime").asLong(),
                    LocalResourceVisibility.valueOf(node.get("visibility").asText()),
                    LocalResourceType.valueOf(node.get("resourceType").asText()));
        } catch (JsonProcessingException e) {
            throw new FlinkException("Error to parse YarnLocalResourceDescriptor from " + desc, e);
        }
    }

    private static boolean validate(JsonNode node) {
        return !node.isNull()
                && node.hasNonNull("resourceKey")
                && node.hasNonNull("path")
                && node.hasNonNull("size")
                && node.hasNonNull("modificationTime")
                && node.hasNonNull("visibility")
                && node.hasNonNull("resourceType");
    }

    static YarnLocalResourceDescriptor fromFileStatus(
            final String key,
            final FileStatus fileStatus,
            final LocalResourceVisibility visibility,
            final LocalResourceType resourceType) {
        checkNotNull(key);
        checkNotNull(fileStatus);
        checkNotNull(visibility);
        return new YarnLocalResourceDescriptor(
                key,
                fileStatus.getPath(),
                fileStatus.getLen(),
                fileStatus.getModificationTime(),
                visibility,
                resourceType);
    }

    @Override
    public String toString() {
        try {
            ObjectNode node = OBJECT_MAPPER.createObjectNode();
            node.put("resourceKey", resourceKey);
            node.put("path", path.toString());
            node.put("size", size);
            node.put("modificationTime", modificationTime);
            node.put("visibility", visibility.toString());
            node.put("resourceType", resourceType.toString());
            return OBJECT_MAPPER.writeValueAsString(node);
        } catch (JsonProcessingException e) {
            LOG.error("Could not serialize YarnLocalResourceDescriptor to String.", e);
            throw new RuntimeException(
                    "Could not serialize YarnLocalResourceDescriptor[%s] to String.", e);
        }
    }

    @Override
    public int hashCode() {
        int result = Objects.hashCode(resourceKey);
        result = 31 * result + Objects.hashCode(path);
        result = 31 * result + Objects.hashCode(size);
        result = 31 * result + Objects.hashCode(modificationTime);
        result = 31 * result + Objects.hashCode(visibility.toString());
        result = 31 * result + Objects.hashCode(resourceType.toString());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        } else if (obj != null && obj.getClass() == YarnLocalResourceDescriptor.class) {
            YarnLocalResourceDescriptor that = (YarnLocalResourceDescriptor) obj;
            return Objects.equals(this.resourceKey, that.resourceKey)
                    && Objects.equals(path, that.path)
                    && Objects.equals(size, that.size)
                    && Objects.equals(modificationTime, that.modificationTime)
                    && Objects.equals(visibility, that.visibility)
                    && Objects.equals(resourceType, that.resourceType);
        }
        return false;
    }

    /**
     * Transforms this local resource descriptor to a {@link LocalResource}.
     *
     * @return YARN resource
     */
    public LocalResource toLocalResource() {
        return Utils.registerLocalResource(path, size, modificationTime, visibility, resourceType);
    }
}
